//*************************************************************
//
//	Linux Driver for CT-3301 (& CT-3301RGB) Frame_Grabber 
//
//	Copyright (C) 2003 Cybertek CO.,
//
//	Used Linux Kernel Version >=(2.4.0)
//
//	Released under the terms of the GPL.
//

#if defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
  #include <linux/modversions.h>
  #define MODVERSIONS
#endif

#include <linux/module.h>
//#include <linux/version.h>
#include <linux/init.h>	// Needed for the macros
#include <linux/mm.h>
#include <linux/fs.h>
#include <linux/pci.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/wait.h>
#include <asm/pgtable.h>
#include <linux/slab.h>
#include <asm/uaccess.h>
#include "ctdv31.h"

static int __init ctdv31_init(void);
static void __exit ctdv31_cleanup(void);
static int ct_open( struct inode*, struct file*);
static int ct_release( struct inode*, struct file*);
static int ct_ioctl( struct inode*,struct file*,UINT,ULONG);
static int ct_mmap(struct file*, struct vm_area_struct*);
static void ct_vma_open(struct vm_area_struct*);
static void ct_vma_close(struct vm_area_struct*);
static ULONG GetPCIConf(PDEVICE_TBL pdevt);
static int SetIntEnable(PDEVICE_TBL pdevt);
static void DisableIntr(PDEVICE_TBL pdevt,UCHAR rst);
static void ctdv31_ISR(int irq,void *dev_id,struct pt_regs *regs);
static void SetVideo_DmaRegister(PDEVICE_TBL pdevt);

static struct file_operations ct_fops = {
        THIS_MODULE,
	(loff_t)NULL,	        // lseek
	(ssize_t)NULL,		// read
	(ssize_t)NULL,		// write
	(int)NULL,		// readdir
	(UINT)NULL,		// poll
	ct_ioctl,		// ioctl
	ct_mmap,		// mmap
	ct_open,		// open
	(int)NULL,		// flush
	ct_release,		// release
	(int)NULL,		// fsync
	(int)NULL,		// fasync
	(int)NULL,		// check_media_change
	(int)NULL,		// lock
        (ssize_t)NULL,     	// readv
        (ssize_t)NULL      	// writev
};

static struct vm_operations_struct ct_vm_ops = {
        ct_vma_open,    // open
        ct_vma_close,   // close
        NULL	        // nopage
};

PDEVICE_TBL pDevTbl;
int MAJOR = 0;
int MEM = 59;
int LOG=0;

MODULE_PARM(MAJOR,"i");
MODULE_PARM(MEM,"i");
MODULE_PARM(LOG,"i");

MODULE_LICENSE("GPL");

static int __init ctdv31_init(void)
{
  int result;
  
  result = register_chrdev(MAJOR, "ctdv31", &ct_fops);
  if(result < 0){
    printk("<1>register_chrdev failed.\n");
    return result;
  }
  else{
    if(MAJOR == 0) MAJOR = result;
    printk("<1>CTDV31 resident(MAJOR=%d).\n",MAJOR);
  }
  pDevTbl = (PDEVICE_TBL)kmalloc(sizeof(DEVICE_TBL)*CT_MAX_BOARD,GFP_KERNEL);  
  if(pDevTbl == NULL){
    printk("<1>memory not available.\n");
    return -ENOMEM;
  }
  memset(pDevTbl,0,sizeof(DEVICE_TBL)*CT_MAX_BOARD);
  if(!pcibios_present()){
    printk("<1>pci bios not present.\n");
    if(pDevTbl) kfree(pDevTbl);
    unregister_chrdev(MAJOR, "ctdv31");
    return -EINVAL;
  }
  return 0;
}

static void __exit ctdv31_cleanup(void)
{
  if(pDevTbl) kfree(pDevTbl);
  if(unregister_chrdev(MAJOR, "ctdv31") != 0)
    printk("<1>cleanup_module failed\n");
  else
    printk("<1>CTDV31 removed.\n");
}

module_init(ctdv31_init);
module_exit(ctdv31_cleanup);

// open & close

static int ct_open( struct inode* inode, struct file* filp)
{
  int brd_id = MINOR(inode->i_rdev);
  PDEVICE_TBL pdevt = &pDevTbl[brd_id];
  int err;
  
  if(brd_id >= CT_MAX_BOARD) return -ENODEV;
  pdevt->brd_id = brd_id;
  // culculate trans_buf area
  pdevt->PhysMemBase = MEM * 0x100000 + CT_MEMORY_SIZE * brd_id;
  if(LOG) printk("<1>PhysMemBase[%d]=%lx.\n",brd_id,pdevt->PhysMemBase);
  filp->private_data = pdevt;
  pdevt->TrsPhysBase = pdevt->PhysMemBase;
  pdevt->parm[1] = 0x3c;	// intr_line
  err = GetPCIConf(pdevt);
  if(err) return err;
  pdevt->IntrLine = pdevt->parm[0] & 0xff;
  pdevt->parm[1] = 0x14;	// iobase
  err = GetPCIConf(pdevt);
  if(err) return err;
  pdevt->IOPortBase = (USHORT)pdevt->parm[0] & 0xfff0;
  spin_lock_init(&pdevt->lock);
  init_waitqueue_head(&pdevt->wque);
  MOD_INC_USE_COUNT;
  return 0;   
}

static int ct_release( struct inode* inode, struct file* filp)
{
  PDEVICE_TBL pdevt = filp->private_data;
  
  if(inb(pdevt->IOPortBase)&CR0_UNDER_WRITE)
    outb((UCHAR)0x00,pdevt->IOPortBase);
  if(inb(pdevt->IOPortBase+1) & CR01_INT_ENABLE)
    DisableIntr(pdevt,CR01_RS_ALL_INT);
  memset(pdevt,0,sizeof(DEVICE_TBL));
  MOD_DEC_USE_COUNT;
  return 0;
}

//
// The ioctl() implementation
//

static int ct_ioctl(struct inode *inode,struct file *filp,UINT cmd,ULONG arg)
{

  int err;
  ULONG tmp_data=0;
  PDEVICE_TBL pdevt = filp->private_data;

  if(_IOC_TYPE(cmd) != CT_IOC_MAGIC) return -EINVAL;
  if(_IOC_NR(cmd) > CT_IOC_MAXNR) return -EINVAL; 
  err = verify_area(VERIFY_WRITE, (long *)arg, sizeof(long) * 4);
  if(err){
    printk("<1>ioctl reference error.\n");
    return -EINVAL;
  }

  copy_from_user(pdevt->parm,(void *)arg,sizeof(long) * 4l);

  switch(cmd){
    case (UINT)CT_GetVersion:
      tmp_data = Driver_version;
      break;
    case (UINT)CT_SetVideo_DmaRegister:
      SetVideo_DmaRegister(pdevt);
      break;
    case (UINT)CT_CaptureStart:
      pdevt->IOControl =
        inl(pdevt->IOPortBase) & 0xf0f6;
      outb((UCHAR)(pdevt->IOControl | (pdevt->parm[1] &
        (CR0_START | CR0_CONT_TRIG))),pdevt->IOPortBase); // Video REG-0
      tmp_data = (ULONG)inw(pdevt->IOPortBase);
      break;
    case (UINT)CT_GetIntStatus:
      spin_lock(&pdevt->lock);
      tmp_data = pdevt->IntStatus;
      pdevt->IntStatus = 0x0;
      spin_unlock(&pdevt->lock);
      break;
    case (UINT)CT_GetPCIConf:
      err = GetPCIConf(pdevt);
      if(err) return -EINVAL;
      tmp_data = pdevt->parm[0];
      break;
    case (UINT)CT_SetIntEnable:
      tmp_data = pdevt->parm[1];
      err = SetIntEnable(pdevt);
      if(err){ put_user(tmp_data,(PULONG)arg); return err;}
      break;
    case (UINT)CT_SleepOn:
      if(inb(pdevt->IOPortBase+1) & CR01_INT_ENABLE){
        interruptible_sleep_on(&pdevt->wque);
        tmp_data = 0;
      }
      else tmp_data = -1;
      break;
    case (UINT)CT_SetTransferMemAddress:
      switch(pdevt->parm[1]){
        default:
          pdevt->TrsPhysBase = tmp_data = 
            pdevt->PhysMemBase;
          break;
        case  1:
          pdevt->TrsPhysBase = tmp_data = pdevt->parm[2];
          break;
      }
      break;
    case (UINT)CT_OUTDW :
      outl(pdevt->parm[2],(UINT)pdevt->parm[1]);
      break;
    case (UINT)CT_OUTW :
      outw((UINT)pdevt->parm[2],(UINT)pdevt->parm[1]);
      break;
    case (UINT)CT_OUTB :
      outb((UCHAR)pdevt->parm[2],(UINT)pdevt->parm[1]);
      break;
    case (UINT)CT_INDW  :
      tmp_data = inl((UINT)pdevt->parm[1]);
      break;
    case (UINT)CT_INW :
      tmp_data = (ULONG)inw((UINT)pdevt->parm[1]);
      break;
    case (UINT)CT_INB :
      tmp_data = (ULONG)inb((UINT)pdevt->parm[1]);
      break;
    case (UINT)CT_IOCHARDRESET :
      break;
    default :
      return -EINVAL;
  }
  put_user(tmp_data,(PULONG)arg);
  return 0;
}

static void SetVideo_DmaRegister(PDEVICE_TBL pdevt)
{
  ULONG odata[7];
  
  (USHORT)pdevt->IOControl = 
    inw(pdevt->IOPortBase) & 0xa0f0;
  switch(pdevt->parm[1]){
    case   0x0:	// 24-bit MIX
      odata[0] = (ULONG)0x8000;
      odata[2] = (ULONG)0xc0f3;	// Ver_Size
      odata[5] = (ULONG)((0x280 + pdevt->parm[3]) * 3);
      odata[6] = (ULONG)0x80000000 + ((pdevt->parm[2] & 0x1) << 29)
        + (BUFFER_OFST - odata[5] * (odata[2] & 0x1ff)) * 2;
      // TRSE & S_M & V_il
      break;
    case   0x1: // 24-bit EVN
      odata[0] = (ULONG)0x8002;
      odata[2] = (ULONG)0xc0f3;
      odata[5] = (ULONG)((0x280 + pdevt->parm[3]) * 3);
      odata[6] = (ULONG)0x80000000 + ((pdevt->parm[2] & 0x1) << 29)
        + (BUFFER_OFST - odata[5] * (odata[2] & 0x1ff)); // TRSE & S_M & V_il
      break;
    case   0x2: // 24-bit ODD
      odata[0] = (ULONG)0x8004;
      odata[2] = (ULONG)0xc0f3;
      odata[5] = (ULONG)((0x280 + pdevt->parm[3]) * 3);
      odata[6] = (ULONG)0x80000000 + ((pdevt->parm[2] & 0x1) << 29)
        + (BUFFER_OFST - odata[5] * (odata[2] & 0x1ff)); // TRSE & S_M & V_il
      break;
    case   0x3: // 24-bit FRM
      odata[0] = (ULONG)0x8006;
      odata[2] = (ULONG)0xc0f3;
      odata[5] = (ULONG)((0x280 + pdevt->parm[3]) * 3);
      odata[6] = (ULONG)0x80000000 + ((pdevt->parm[2] & 0x1) << 29)
        + (BUFFER_OFST/2 - odata[5] * (odata[2] & 0x1ff)); // TRSE & S_M & V_il
      break;
    case   0x4: // 24-bit PRG
      odata[0] = (ULONG)0xc000;
      odata[2] = (ULONG)0xc1e6;
      odata[5] = (ULONG)((0x280 + pdevt->parm[3]) * 3);
      odata[6] = (ULONG)0x80000000 + ((pdevt->parm[2] & 0x1) << 29)
        + BUFFER_OFST - odata[5] * (odata[2] & 0x1ff); // TRSE & S_M & V_il
      break;
    case 0x5: // 32-bit MIX
      odata[0] = (ULONG)0x8000;
      odata[2] = (ULONG)0xe0f3;
      odata[5] = (ULONG)((0x280 + pdevt->parm[3]) * 4);
      odata[6] = (ULONG)0x80000000 + ((pdevt->parm[2] & 0x1) << 29)
        + (BUFFER_OFST - odata[5] * (odata[2] & 0x1ff)) * 2;
      // TRSE & S_M & V_il
      break;
    case 0x6: // 32-bit EVN
      odata[0] = (ULONG)0x8002;
      odata[2] = (ULONG)0xe0f3;
      odata[5] = (ULONG)((0x280 + pdevt->parm[3]) * 4);
      odata[6] = (ULONG)0x80000000 + ((pdevt->parm[2] & 0x1) << 29)
        + (BUFFER_OFST - odata[5] * (odata[2] & 0x1ff)); // TRSE & S_M & V_il
      break;
    case 0x7: // 32-bit ODD
      odata[0] = (ULONG)0x8004;
      odata[2] = (ULONG)0xe0f3;
      odata[5] = (ULONG)((0x280 + pdevt->parm[3]) * 4);
      odata[6] = (ULONG)0x80000000 + ((pdevt->parm[2] & 0x1) << 29)
        + (BUFFER_OFST - odata[5] * (odata[2] & 0x1ff)); // TRSE & S_M & V_il
      break;
    case 0x8: // 32-bit FRM
      odata[0] = (ULONG)0x8006;
      odata[2] = (ULONG)0xe0f3;
      odata[5] = (ULONG)((0x280 + pdevt->parm[3]) * 4);
      odata[6] = (ULONG)0x80000000 + ((pdevt->parm[2] & 0x1) << 29)
        + (BUFFER_OFST/2 - odata[5] * (odata[2] & 0x1ff)); // TRSE & S_M & V_il
      break;
    case 0x9: // 32-bit PRG
      odata[0] = (ULONG)0xc000;
      odata[2] = (ULONG)0xe1e6;
      odata[5] = (ULONG)((0x280 + pdevt->parm[3]) * 4);
      odata[6] = (ULONG)0x80000000 + ((pdevt->parm[2] & 0x1) << 29)
        + (BUFFER_OFST - odata[5] * (odata[2] & 0x1ff)); // TRSE & S_M & V_il
      break;
    default:
      odata[0] = 0;
      break;
  }
  if(!odata[0]) return;
  pdevt->IOControl |= odata[0];
  outw((USHORT)odata[2],pdevt->IOPortBase+OFS_HVSIZE);	// Video REG-2
  outl(pdevt->TrsPhysBase,pdevt->IOPortBase+OFS_MADDRESS);
        // Memory Start Register
  if(LOG) printk("<1>trs_address=%lx.\n",pdevt->TrsPhysBase);
  outw((USHORT)odata[5],pdevt->IOPortBase+OFS_HWORD_CNT);
	// H-Word Count
  outl(odata[6],pdevt->IOPortBase+OFS_VILEAVE);
	// TRSE,S_M,V-Word Interleave
  outw((USHORT)pdevt->IOControl,pdevt->IOPortBase);
	// Video REG-0
}

static int SetIntEnable(PDEVICE_TBL pdevt)
{
  USHORT pData;
  int rslt=0;

  spin_lock(&pdevt->lock);
  if(pdevt->parm[1] == 1){	// Enable Intr
    rslt = 
      request_irq(pdevt->IntrLine,ctdv31_ISR,SA_INTERRUPT | SA_SHIRQ,
        "ctdv31",pdevt);
    if(rslt){
      printk("<1>can't get %ld-irq(%x).\n",
        pdevt->IntrLine,rslt);
      return rslt;
    }
    pData = inb(pdevt->IOPortBase+1) & CR01_READ_MASK;
    outb(pData|CR01_MSE|CR01_INT_ENABLE|CR01_RS_ALL_INT,
      pdevt->IOPortBase+1);
  }
  else DisableIntr(pdevt,0);
  spin_unlock(&pdevt->lock);
  return 0;
}

static void DisableIntr(PDEVICE_TBL pdevt,UCHAR rst)
{
  USHORT pData;

  pData = inb(pdevt->IOPortBase+1) & CR01_READ_MASK;
  outb((UCHAR)((pData & ~CR01_INT_ENABLE) | rst | CR01_MSE),
    pdevt->IOPortBase+1);
  free_irq(pdevt->IntrLine,pdevt);
  wake_up_interruptible(&pdevt->wque);
}

static void ctdv31_ISR(int irq,void *dev_id,struct pt_regs *regs)
{
  UCHAR intd;
  int i;
  PDEVICE_TBL pdevt = (PDEVICE_TBL)dev_id;

  intd = (UCHAR)inb(pdevt->IOPortBase+1);
  if(!(intd & CR01_INT_ENABLE)) return;
  intd &= CR01_INT_MASK;
  if(!intd) return;
  for(i=0;(i<4)&&intd;i++){
    pdevt->IntStatus |= (ULONG)1 << (intd - 1);
    // current int reset
    outb((UCHAR)CR01_RS_CUR_INT,pdevt->IOPortBase+1);
    intd = (UCHAR)inb(pdevt->IOPortBase+1) & CR01_INT_MASK;
  }
  wake_up_interruptible(&pdevt->wque);
}

static ULONG GetPCIConf(PDEVICE_TBL pdevt)
{
  struct pci_dev *dev=NULL;
  int i;
  
  if(dev != NULL) dev = NULL;
  for(i=0;i<=pdevt->brd_id;i++)
    dev = pci_find_device((UINT)CT_VENDOR_ID,(UINT)CT_DEVICE_ID,dev);
  if(dev == NULL) return -EINVAL;
  pci_read_config_dword(dev,(UCHAR)pdevt->parm[1],(u32 *)&pdevt->parm[0]);
  return 0;
}

//
//	The mmap() implementation
//

static int ct_mmap(struct file * filp, struct vm_area_struct * vma)
{
  PDEVICE_TBL pdevt = filp->private_data;
  ULONG msize,offset;
  
  offset = vma->vm_pgoff << PAGE_SHIFT;
  offset += pdevt->TrsPhysBase;
  msize = vma->vm_end - vma->vm_start;
  if(msize > CT_MEMORY_SIZE) msize = CT_MEMORY_SIZE;
//  pgprot_val(vma->vm_page_prot) |= (_PAGE_PCD | _PAGE_PWT); // caching disable 
#if LINUX_VERSION_CODE < KERNEL_VER(2,4,20)
  if(remap_page_range(vma->vm_start,offset,msize,vma->vm_page_prot))
    return -EAGAIN;
#else
 if(remap_page_range(vma,vma->vm_start,offset,msize,vma->vm_page_prot))
    return -EAGAIN;
#endif
  vma->vm_ops = &ct_vm_ops;
  vma->vm_flags |= VM_RESERVED;
  ct_vma_open(vma);
  return 0;
}

static void ct_vma_open(struct vm_area_struct *vma)
{
  MOD_INC_USE_COUNT;
}

static void ct_vma_close(struct vm_area_struct *vma)
{
  MOD_DEC_USE_COUNT;
}

